/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.rest.session;

import java.io.IOException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.core.Response;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.User;
import gov.vha.isaac.ochre.api.UserCache;
import gov.vha.isaac.ochre.api.UserRole;
import gov.vha.isaac.ochre.api.util.UuidT5Generator;

/**
 * 
 * {@link UserServiceUtils}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class UserServiceUtils {
	public static final String TEST_JSON1 = "{\"roles\":["
	+ "{\"id\":19990,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19991,\"name\":\"editor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19992,\"name\":\"reviewer\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19993,\"name\":\"administrator\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19994,\"name\":\"manager\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19995,\"name\":\"vuid_requestor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19996,\"name\":\"approver\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ "],\"token_parsed?\":true,\"user\":\"TEST\",\"type\":\"ssoi\",\"id\":10005}";
	public static final String TEST_JSON2 = "{\"roles\":["
	+ "{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ "],\"token_parsed?\":true,\"user\":\"DNS   \",\"type\":\"ssoi\",\"id\":10005}";
	public static final String TEST_JSON3 = "{\"roles\":["
	+ "{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19991,\"name\":\"editor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ "],\"token_parsed?\":true,\"user\":\"DNS   \",\"type\":\"ssoi\",\"id\":10005}";

	private UserServiceUtils() {}

	/**
	 * 
	 * Attempt to construct a user from a string of the following format:
	 * 
	 * {name}:{role1}[{,role2}[{,role3}[...]]]
	 * 
	 * @param arg
	 * @return
	 */
	static Optional<String> constructTestUser(String arg) {
		try {
			String[] components = arg.split(":");
	
			String name = null;
			Set<UserRole> roles = new HashSet<>();
			if (components.length == 2) {
				if (components[0].matches("[A-Za-z][A-Za-z0-9_]*")) {
					name = components[0].trim();
	
					String[] roleStrings = components[1].split(",");
	
					for (int i = 0; i < roleStrings.length; ++i) {
						roles.add(UserRole.safeValueOf(roleStrings[i].trim()).get());
					}
				}
			}
			
			if (name != null && name.length() > 0 && roles.size() > 0) {
				StringBuilder builder = new StringBuilder();
				builder.append("{\"roles\":[");
				boolean addedRole = false;
				for (UserRole role : roles) {
					if (addedRole) {
						builder.append(",");
					}
					builder.append("{\"id\":19990,\"name\":\"" + role.toString() 
					+ "\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}");
	
					addedRole = true;
				}
				
				builder.append("],\"token_parsed?\":true,\"user\":\"" + name + "\",\"type\":\"ssoi\",\"id\":10005}");
				
				return Optional.of(builder.toString());
			}
		} catch (Exception e) {
			// ignore
		}
	
		return Optional.empty();
	}

	public static Optional<User> getUserFromTestToken(String ssoToken) throws JsonParseException, JsonMappingException, IOException {
		String jsonToUse = null;
	
		Optional<String> createdJson = constructTestUser(ssoToken);
		if (createdJson.isPresent()) {
			jsonToUse = createdJson.get();
		} else {
			if (ssoToken.equals(TEST_JSON1)) {
				jsonToUse = TEST_JSON1;
			} else if (ssoToken.equals(TEST_JSON2)) {
				jsonToUse = TEST_JSON2;
			} else if (ssoToken.equals(TEST_JSON3)) {
				jsonToUse = TEST_JSON3;
			} else if (ssoToken.equals("TEST_JSON1") || ssoToken.equals("TEST")) {
				jsonToUse = TEST_JSON1;
			} else if (ssoToken.equals("TEST_JSON2")) {
				jsonToUse = TEST_JSON2;
			} else if (ssoToken.equals("TEST_JSON3")) {
				jsonToUse = TEST_JSON3;
			} else {
				// Either a real SSO token or custom JSON
				jsonToUse = ssoToken;
			}
		}
	
		/*
		 * Example URL for get_roles_by_token
		 * URL url = new URL("https://DNS                     get_roles_by_token.json?token=" + token);
		 */
		User newUser = null;
		try {
			newUser = UserServiceUtils.getUserFromJson(jsonToUse);
		} catch (Exception e) {
			try {
				newUser = UserServiceUtils.getUserFromJson(TEST_JSON1);
			} catch (IOException e1) {
				throw new RuntimeException(e1);
			}
		}
		
		LookupService.getService(UserCache.class).put(newUser);
	
		return Optional.of(newUser);
	}

	private static void testToken(String token) {
		PrismeIntegratedUserService service = LookupService.getService(PrismeIntegratedUserService.class);

		PrismeServiceUtils.getPrismeProperties().setProperty(
				"prisme_roles_by_token_url",
				"https://DNS              :PORT");
		PrismeServiceUtils.getPrismeProperties().setProperty(
				"prisme_roles_by_token_url",
				"https://DNS              :PORT");
		try {
			URL url = new URL(service.getPrismeRolesByTokenUrl());

			Client client = ClientBuilder.newClient();
			
			Response response = client.target(PrismeServiceUtils.getTargetFromUrl(url))
					.path(url.getPath())
					.queryParam("token", token)
					.request().get();

			String responseJson = response.readEntity(String.class);
			
			ObjectMapper mapper = new ObjectMapper();
			Map<?, ?> map = null;
			try {
				map = mapper.readValue(responseJson, Map.class);
			} catch (Exception e) {
				throw new Exception("Failed parsing json \"" + responseJson);
			} finally {
				client.close();
			}

			System.out.println("Output: \"" + map + "\"");
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public static void main(String...argv) throws Exception {
//		 * Example URL for get_roles_by_token
//		 * URL url = new URL("https://DNS                    ?token=" + token);
		
		//{"user":"joel.kniaz@vetsez.com","roles":["read_only","super_user","administrator","vuid_requestor"],"token":"%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC7%5CxF2%5CxE8%5CxA5%5CxD8%5CxE3t%5CxFFUK%22%2C+%22%2CJ%5Cx83%5CxA3%5Cx13k%5Cx96%5CxFC%5CxE6%5CxF3%5CxCF%5CxF2%7C%5CxB8MK%22%2C+%224%5Cf%5Cx8C%5CxBA%5Cx1Ft%5CxDD%5CxB5%5CxA4%5CxB8%5CxC0%5CxE9Q%5CxAB%5CnK%22%2C+%22z%5D%5Cx83%5CxAFT%7B%5Cx9C%5CxB3%5CxE8%5CxAC%5CxA7%5Cx95%5Cx17%5CxDBiL%22%5D"}
		//final String         ssoToken = "%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx91%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFX%7C%5Cx96%5CxA8%5CxA3%5CxA2%5CxC4%5CxB1%3D%5CxFF%5Cx01K%22%2C+%22oC%5Cx83%5CxF7%40%3A%5Cx94%5CxAC%5CxAF%5CxB6%5CxE1%5CxF4c%5CxB8%5CbK%22%2C+%22+M%5Cx89%5CxB8Xe%5CxF9%5CxD4%5CxC0%5CxDB%5CxAB%5Cx99%5Ce%5CxD7e%40%22%5D";

		//{"user":"readonly@readonly.com","roles":["read_only"],"token":"%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC6%5CxF2%5CxE8%5CxA5%5CxD8%5CxE3t%5CxFFUK%22%2C+%22%2CJ%5Cx83%5CxA3%5Cx13k%5Cx96%5CxFC%5CxE6%5CxF3%5CxCF%5CxF2%7C%5CxB8MK%22%2C+%224%5Cf%5Cx94%5CxB0%5Ce%7C%5Cx9C%5CxB0%5CxA6%5CxA8%5CxE1%5CxE1t%5CxBC%5CvK%22%2C+%22a%40%5Cx8A%5CxACT%7B%5Cx9C%5CxB3%5CxE8%5CxAC%5CxA7%5Cx95%5Cx17%5CxDBiL%22%5D"}
		//final String readOnlySsoToken = "%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC6%5CxF2%5CxE8%5CxA5%5CxD8%5CxE3t%5CxFFUK%22%2C+%22%2CJ%5Cx83%5CxA3%5Cx13k%5Cx96%5CxFC%5CxE6%5CxF3%5CxCF%5CxF2%7C%5CxB8MK%22%2C+%224%5Cf%5Cx94%5CxB0%5Ce%7C%5Cx9C%5CxB0%5CxA6%5CxA8%5CxE1%5CxE1t%5CxBC%5CvK%22%2C+%22a%40%5Cx8A%5CxACT%7B%5Cx9C%5CxB3%5CxE8%5CxAC%5CxA7%5Cx95%5Cx17%5CxDBiL%22%5D";

		// Dan (read_only)
		//final String ssoToken = "%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC0%5CxF2%5CxE8%5CxA5%5CxD8%5CxE3t%5CxFFUK%22%2C+%22%2CJ%5Cx83%5CxA3%5Cx13k%5Cx96%5CxFC%5CxE6%5CxF3%5CxCF%5CxF2%7C%5CxB8MK%22%2C+%224%5Cf%5Cx85%5CxA7%5Cx13k%5CxB3%5CxBD%5CxB8%5CxB8%5CxD2%5CxBDr%5CxB2%5Cx02K%22%2C+%22%2CS%5CxE8%5CxDBt%5Cx16%5CxFD%5CxD0%5CxC4%5CxDF%5CxAF%5Cx9D%5Cx1F%5CxD3aD%22%5D";
		// Greg (super_user)
		//final String ssoToken = "%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx91%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFX%7C%5Cx96%5CxA8%5CxA3%5CxA2%5CxC4%5CxB1%3D%5CxFF%5Cx01K%22%2C+%22oC%5Cx83%5CxF7%40%3A%5Cx94%5CxAC%5CxAF%5CxB6%5CxE1%5CxF4c%5CxB8%5CbK%22%2C+%22+M%5Cx89%5CxB8Xe%5CxF9%5CxD4%5CxC0%5CxDB%5CxAB%5Cx99%5Ce%5CxD7e%40%22%5D";

		// read_only (read_only)
		//final String ssoToken = "%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx91%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFX%7C%5Cx96%5CxA8%5CxA3%5CxA2%5CxC4%5CxB1%3D%5CxFF%5Cx01K%22%2C+%22oC%5Cx83%5CxF7%40%3A%5Cx94%5CxAC%5CxAF%5CxB6%5CxE1%5CxF4c%5CxB8%5CbK%22%2C+%22+M%5Cx89%5CxB8Xe%5CxF9%5CxD4%5CxC0%5CxDB%5CxAB%5Cx99%5Ce%5CxD7e%40%22%5D";
	
		//final String urlStr = "https://DNS              :PORT?id=cris@cris.com&password=cris@cris.com&token=" + ssoToken;
		//final String url = "https://DNS              :PORT?token=" + ssoToken;

		PrismeUserService PRISME_USER_SERVICE = LookupService.getService(PrismeIntegratedUserService.class);

		testToken(PRISME_USER_SERVICE.getToken("joel.kniaz@vetsez.com", "joel.kniaz@vetsez.com"));
		testToken(PRISME_USER_SERVICE.getToken("readonly@readonly.com", "readonly@readonly.com"));
	}

	static User getUserFromJson(String jsonToUse) throws JsonParseException, JsonMappingException, IOException {
		ObjectMapper mapper = new ObjectMapper();
		Map<?, ?> map = null;
		map = mapper.readValue(jsonToUse, Map.class);
		
		String userName = (String)map.get("user");
		Set<UserRole> roleSet = new HashSet<>();
		Collection<?> roles = (Collection<?>)map.get("roles");
		for (Object roleMapObject : roles) {
			Map<?,?> roleMap = (Map<?,?>)roleMapObject;
			String roleName = (String)roleMap.get("name");
			
			roleSet.add(UserRole.safeValueOf(roleName).get());
		}
		
		final UUID uuidFromUserFsn = UserServiceUtils.getUuidFromUserName(userName);;
	
		User newUser = new User(userName, uuidFromUserFsn, roleSet);
		
		return newUser;
	}

	static Set<UserRole> getAllRolesFromUrl(URL url) throws JsonParseException, JsonMappingException, IOException {
		String jsonResultString = PrismeServiceUtils.getResultJsonFromPrisme(PrismeServiceUtils.getTargetFromUrl(url), url.getPath());
		
		Set<UserRole> roles = new HashSet<>();
		
		ObjectMapper mapper = new ObjectMapper();
		Object returnedObject = mapper.readValue(jsonResultString, List.class);
		
		for (Object roleFromPrisme : (List<?>)returnedObject) {
			roles.add(UserRole.valueOf(roleFromPrisme.toString()));
		}
		
		return Collections.unmodifiableSet(roles);
	}
	static String getUserSsoTokenFromUrl(URL url, String id, String password) throws Exception {
		Map<String, String> params = new HashMap<>();	
		params.put("id", id);
		params.put("password", password);
		String jsonResultString = PrismeServiceUtils.getResultJsonFromPrisme(PrismeServiceUtils.getTargetFromUrl(url), url.getPath(), params);
		
		ObjectMapper mapper = new ObjectMapper();
		Map<?, ?> map = null;
		try {
			map = mapper.readValue(jsonResultString, Map.class);
		} catch (Exception e) {
			throw new Exception("Failed parsing json \"" + jsonResultString);
		}
		//System.out.println("Output for id=\"" + id + "\", password=\"" + password + "\": \"" + map + "\"");
		
		return (String)map.get("token");
	}

	static UUID getUuidFromUserName(String userName) {
		return UuidT5Generator.get(MetaData.USER.getPrimordialUuid(), userName);
	}

	static Optional<User> getUserFromUrl(URL url, String ssoToken) throws JsonParseException, JsonMappingException, IOException {
//		/*
//		 * Example URL for get_roles_by_token
//		 * URL url = new URL("https://DNS                     ?token=" + token);
//		 */
//		/*
//		 * Example SSO Token
//		 * %5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx94%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFXk%5Cx80%5CxB1%5CxA3%5CxF3%5Cx8D%5CxB1%5Cx7F%5CxBC%5Cx02K%22%2C+%22k%5Cf%5CxDC%5CxF7%2CP%5CxB2%5Cx97%5Cx99%5Cx99%5CxE0%5CxE1%7C%5CxBF%5Cx1DK%22%2C+%22J%5Cf%5Cx9B%5CxD8w%5Cx15%5CxFE%5CxD3%5CxC7%5CxDC%5CxAC%5Cx9E%5Cx1C%5CxD0bG%22%5D
//		 */
//		//String json = "{\"roles\":[{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}],\"token_parsed?\":true,\"user\":\"DNS   \",\"type\":\"ssoi\",\"id\":10005}";

		Map<String, String> params = new HashMap<>();	
		params.put("token", URLEncoder.encode(ssoToken, "UTF-8"));
		String jsonResultString = PrismeServiceUtils.getResultJsonFromPrisme(PrismeServiceUtils.getTargetFromUrl(url), url.getPath(), params);
		
		return Optional.of(UserServiceUtils.getUserFromJson(jsonResultString));
	}
}
